CSSwhat = require "CSSwhat"

#functions that make porting the library to another DOM easy
isElement = (elem = "elem") ->
	"""
	(#{elem}.type === 'tag' || #{elem}.type === 'style' || #{elem}.type === 'script')
	"""
getChildren = (elem = "elem") -> "#{elem}.children"
getParent = (elem = "elem") -> "#{elem}.parent"
getAttributeValue = (elem = "elem", name) -> "#{elem}.attribs[#{quote name}]"
hasAttrib = (elem = "elem", name) -> "(#{elem}.attribs && #{quote name} in #{elem}.attribs)"
getName = (elem = "elem") -> "#{elem}.name"
getText = (elem = "elem") ->
	"""
	(function getText(node){
		var text = "",
			childs = #{getChildren "node"};
	
		if(!childs) return text;
	
		for(var i = 0, j = childs.length; i < j; i++){
			if(#{isElement "childs[i]"}) text += getText(childs[i]);
			else text += childs[i].data;
		}
	
		return text;
	})(#{elem})
	"""

###
	helper methods
###

#returns the siblings of an element
getSiblings = (elem) ->
	"""
	#{getParent elem} && #{getChildren getParent elem}
	"""

#finds the position of an element among its siblings
getIndex = (elem) ->
	"""
	var siblings = #{getSiblings elem},
	    pos = -1;
	
	if(siblings){
		for(var i = 0, j = siblings.length; i < j; i++){
			if(#{isElement "siblings[i]"}) pos++;
			if(siblings[i] === #{elem}) break;
		}
		if(i === j) pos = -1;
	}
	"""

getAttribFunc = (name, value) ->
	(next) -> checkAttrib next, name, value, false

checkAttrib = (next, name, value, ignoreCase) ->
	"""
	if(#{hasAttrib "elem", name} && #{getAttributeValue "elem", name}
		#{if ignoreCase then ".toLowerCase()" else ""} === #{quote value}){
		#{next}
	}
	"""

rootFunc = "return true"
trueFunc = "return true;"
falseFunc = "return false;"

quote = (str) -> "'#{str.replace(/'/g, "\\'")}'" #escape all single quotes

# we need unique names for saving the elem variable

possibleChars = []
	.concat (String.fromCharCode c for c in [48...57]) #0-9
	.concat (String.fromCharCode c for c in [65...91]) #A-Z
	.concat (String.fromCharCode c for c in [97...123]) #a-z
	.concat "_", "$"


lastUniqueName = ""
uniqNameCounter = 0

getUniqueName = () ->
	lastUniqueName = "_$__" #a unique prefix to avoid collisions
	counter = uniqNameCounter
	uniqNameCounter += 1
	while counter > 0
		if counter >= possibleChars.length
			counter -= possibleChars.length-1
			lastUniqueName += possibleChars[possibleChars.length-1]
		else
			lastUniqueName += possibleChars[counter]
			counter = 0
	lastUniqueName

###
	all available rules
###
generalRules = {
	tag: (next, {name}) ->
		"""
		if(#{getName()} === #{quote name}){ #{next} }
		"""
	
	#traversal
	descendant: (next) ->
		"""
		var #{getUniqueName ""} = elem;
		while(elem = #{getParent "elem"}){
			#{next}
		}
		elem = #{lastUniqueName};
		"""
	child: (next) ->
		"""
		var #{getUniqueName ""} = elem;
		if(elem = #{getParent "elem"}){
			#{next}
		}
		elem = #{lastUniqueName};
		"""
	sibling: (next) ->
		sibling = getUniqueName()
		elemName = getUniqueName()
		"""
		var #{sibling} = #{getSiblings "elem"},
		    #{elemName} = elem;

		if(#{sibling}){
			for(var i = 0, j = #{sibling}.length; i < j; i++){
				elem = #{sibling}[i];
				if(#{isElement "elem"}){
					if(#{sibling}[i] === #{elemName}) break;
					#{next}
				}
			}
		}
		elem = #{elemName};
		"""
	adjacent: (next) ->
		"""
		var siblings = #{getSiblings "elem"},
		    #{getUniqueName ""} = elem;
		if(siblings){
			var lastElement;
			for(var i = 0, j = siblings.length; i < j; i++){
				if(#{isElement "siblings[i]"}){
					if(siblings[i] === elem){
						if(lastElement){
							elem = lastElement;
							#{next}
						}
						break;
					}
					lastElement = siblings[i];
				}
			}
		}
		elem = #{lastUniqueName};
		"""
	
	universal: (next) -> if next is rootFunc then trueFunc else next
	
	#attributes
	id: (next, {value}) -> attributeRules.equals(next, "id", value, false);
	class: (next, {value}) -> attributeRules.element(next, "class", value, false);
	attribute: (next, {ignoreCase, action, name, value}) ->
		value = value.toLowerCase() if ignoreCase
		attributeRules[action] next, name, value, ignoreCase
	
	#pseudos
	pseudo: (next, {name, data: subselect}) ->
		if filters[name]? then filters[name] next, subselect
		else if pseudos[name]?
			"""
			if(pseudos[#{quote name}](elem, #{quote subselect})){
				#{next}
			}
			"""
		else next #throw error?
}

###
	attribute selectors
###

reChars = /[-[\]{}()*+?.,\\^$|#\s]/g; #https://github.com/slevithan/XRegExp/blob/master/src/xregexp.js#L469

wrapReRule = (pre, post) ->
	(next, name, value, ignoreCase) ->
		"""
		if(
			#{hasAttrib "elem", name} &&
			/#{pre + value.replace(reChars, "\\$&") + post}/#{if ignoreCase then "i" else ""}.test(#{getAttributeValue "elem", name})
		){
			#{next}
		} 
		"""

attributeRules =
	equals: checkAttrib
	hyphen: wrapReRule "^", "(?:$|-)"
	element: wrapReRule "(?:^|\\s)", "(?:$|\\s)"
	exists: (next, name) -> "if(#{hasAttrib "elem", name}){ #{next} }"
	start: (next, name, value, ignoreCase) ->
		"""
		if(
			#{hasAttrib "elem", name} &&
		    #{getAttributeValue "elem", name}.substr(0, #{value.length})
		    	#{if ignoreCase then ".toLowerCase()" else ""} === #{quote value}
		){
			#{next}
		}
		"""
	end: (next, name, value, ignoreCase) ->
		"""
		if(
			#{hasAttrib "elem", name} &&
		     #{getAttributeValue "elem", name}.substr(#{-value.length})
		     	#{if ignoreCase then ".toLowerCase()" else ""} === #{quote value}
		){
			#{next}
		}
		"""
	any: (next, name, value, ignoreCase) ->
		"""
		if(
			#{hasAttrib "elem", name} &&
		    #{getAttributeValue "elem", name}#{if ignoreCase then ".toLowerCase()" else ""}
		    	.indexOf(#{quote value}) >= 0
		){
			#{next}
		}
		"""
	not: (next, name, value, ignoreCase) ->
		if value is ""
			"""
			if(
				#{hasAttrib "elem", name} &&
				#{getAttributeValue "elem", name} !== ""
			){
				#{next}
			}
			"""
		else
			"""
			if(
				!#{hasAttrib "elem", name} ||
				#{getAttributeValue "elem", name}
					#{if ignoreCase then ".toLowerCase()" else ""} !== #{quote value}
			){
				#{next}
			}
			"""


###
	pseudo selectors
	
	---
	
	they are available in two forms:
	* filters called when the selector 
	  is compiled and return a function
	  that needs to return next()
	* pseudos get called on execution
	  they need to return a boolean
###

###
	returns a function that checks if an elements index matches the given rule
	highly optimized to return the fastest solution
###

re_nthElement = /^([+\-]?\d*n)?\s*([+\-])?\s*(\d)?$/;

getNCheck = (formula) ->
	#parse the formula
	#b is lowered by 1 as the rule uses index 1 as the start
	formula = formula.trim().toLowerCase()
	if formula is "even"
		a = 2
		b = -1
	else if formula is "odd"
		a = 2;
		b = 0;
	else
		formula = formula.match re_nthElement
		return null unless formula? #rule couldn't be parsed
		if formula[1]
			a = parseInt formula[1], 10
			if not a
				if formula[1].charAt(0) is "-" then a = -1
				else a = 1
		else a = 0;
		if formula[3] then b = parseInt((formula[2] || "") + formula[3], 10) - 1
		else b = -1

	#when b <= 0, a*n won't be possible for any matches when a < 0
	#besides, the specification says that no element is matched when a and b are 0
	return falseFunc if b < 0 and a <= 0

	#when b <= 0 and a === 1, they match any element
	return trueFunc if b < 0 and a is 1

	#when a is in the range -1..1, it matches any element (so only b is checked)
	"pos - #{b} <= 0" if a is -1
	"pos - #{b} >= 0" if a is 1
	"pos === b" if a is 0

	#when a > 0, modulo can be used to check if there is a match
	#TODO: needs to be checked
	"pos >= 0 && (pos -= #{b}) >= 0 && (pos % #{a}) === 0" if a > 1

	a *= -1 #make a positive
	"pos >= 0 && (pos -= #{b}) >= 0 && (pos % #{a}) === 0 && pos/#{a} < #{b}"


filters = {
	not: (next, select) ->
		func = parse(select);

		if func is falseFunc
			trueFunc if next is rootFunc
			else next
		else if func in [trueFunc, rootFunc] then falseFunc
		else "if(!(function(elem){#{func}})(elem)){ #{next} }"

	contains: (next, text) ->
		"if(#{getText "elem"}.indexOf(#{quote text}) >= 0){ #{next} }"

	has: (next, select) ->
		func = parse select
		if func == trueFunc or func == rootFunc then next
		else if func == falseFunc then falseFunc
		
		else
			"""
			if((function proc(elem){
				var children = #{getChildren "elem"};
				if(!children) return;
				for(var i = 0, j = children.length; i < j; i++){
					if(!#{isElement "children[i]"}) continue;
					if((function(elem){ #{func} })(children[i])) return true;
					if(proc(children[i])) return true;
				}
			})(elem)){ #{next} }
			"""
	root: (next) -> "if(!#{getParent "elem"}){ #{next} }"
	empty: (next) ->
		"""
		if(!#{getChildren "elem"} || #{getChildren "elem"}.length === 0){
		#{next}
		}
		"""
	parent: (next) -> #:parent is the inverse of :empty
		"""
		if(#{getChildren "elem"} && #{getChildren "elem"}.length !== 0){
		#{next}
		}
		"""
	
	#location specific methods
	#first- and last-child methods return as soon as they find another element
	"first-child": (next) ->
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var i = 0, j = siblings.length; i < j; i++){
				if(#{isElement "siblings[i]"}){
					if(siblings[i] === elem){
						#{next}
					}
					break;
				}
			}
		}
		"""
	"last-child": (next) ->
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var i = siblings.length-1; i >= 0; i--){
				if(#{isElement "siblings[i]"}){
					if(siblings[i] === elem){
						#{next}
					}
					break;
				}
			}
		}
		"""
	"first-of-type": (next) ->
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var i = siblings.length-1; i >= 0; i--){
				if(#{getName "siblings[i]"} === #{getName()}){
					if(siblings[i] === elem){
						#{next}
					}
					break;
				}
			}
		}
		"""
	"last-of-type": (next) ->
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var i = siblings.length-1; i >= 0; i--){
				if(#{getName "siblings[i]"} === #{getName()}){
					if(siblings[i] === elem){
						#{next}
					}
					break;
				}
			}
		}
		"""
	"only-of-type": (next) ->
		"""
		var siblings = #{getSiblings "elem"};
		#{getUniqueName()}: if(siblings){
			for(var i = 0, j = siblings.length; i < j; i++){
				if(siblings[i] === elem) continue;
				if(#{getName "siblings[i]"} === #{getName()}) break #{lastUniqueName};
			}
			
			#{next}
		}
		"""
	"only-child": (next) ->
		"""
		var siblings = #{getSiblings "elem"};
		#{getUniqueName()}: if(siblings){
			if(siblings.length !== 1){
				for(var i = 0, j = siblings.length; i < j; i++){
					if(#{isElement "siblings[i]"} && siblings[i] !== elem) break #{lastUniqueName};
				}
			}
			
			#{next}
		}
		"""

	#nth-element selectors
	"nth-child": (next, rule) ->
		rule = getNCheck rule

		return next if rule is null
		return func if rule is falseFunc
		if rule is trueFunc
			return rule if next is rootFunc else next

		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var pos = 0, i = 0, j = siblings.length; i < j; i++){
				if(siblings[i] === elem){
					if(#{rule}){ #{next} }
				}
				if(#{isElement "siblings[i]"}) pos++;
			}
		}
		"""
	"nth-last-child": (next, rule) ->
		rule = getNCheck rule

		return next if rule is null
		return func if rule is falseFunc
		if rule is trueFunc
			return rule if next is rootFunc else next
		
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var pos = 0, i = siblings.length - 1; i >= 0; i--){
				if(siblings[i] === elem){
					if(#{rule}){ #{next} }
				}
				if(#{isElement "siblings[i]"}) pos++;
			}
		}
		"""
	"nth-of-type": (next, rule) ->
		rule = getNCheck rule

		return next if rule is null
		return func if rule is falseFunc
		if rule is trueFunc
			return rule if next is rootFunc else next
		
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var pos = 0, i = 0, j = siblings.length; i < j; i++){
				if(siblings[i] === elem){
					if(#{rule}){ #{next} }
				}
				if(#{getName "siblings[i]"} === #{getName()}) pos++;
			}
		}
		"""
	"nth-last-of-type": (next, rule) ->
		rule = getNCheck rule

		return next if rule is null
		return func if rule is falseFunc
		if rule is trueFunc
			return rule if next is rootFunc else next
		
		"""
		var siblings = #{getSiblings "elem"};
		if(siblings){
			for(var pos = 0, i = siblings.length-1; i >= 0; i--){
				if(siblings[i] === elem){
					if(#{rule}){ #{next} }
				}
				if(#{getName "siblings[i]"} === #{getName()}) pos++;
			}
		}
		"""
	
	#forms
	selected: (next) ->
		"""
		#{getUniqueName()}: {
			if(!(#{hasAttrib "elem", "selected"})){
				if(#{getName getParent "elem"} !== 'option'){
					break #{lastUniqueName};
				}
				var siblings = #{getSiblings "elem"};
				if(!siblings) break #{lastUniqueName};
				for(var i = 0, j = siblings.length; i < j; i++){
				    if(#{isElement "siblings[i]"}){
				    	if(siblings[i] !== elem) break #{lastUniqueName};
				    	break;
				    }
				}
			}
			#{next}
		}	
		"""
	
	#jQuery extensions
	header: (next) ->
		"""
		var name = #{getName()};
		if(
		    name === "h1" ||
		    name === "h2" ||
		    name === "h3" ||
		    name === "h4" ||
		    name === "h5" ||
		    name === "h6"
		){ #{next} }
		"""
	button: (next) ->
		"""
		if(
			#{getName()} === "button" ||
		    #{getName()} === "input" &&
		    #{hasAttrib "elem", "type"} &&
		    #{getAttributeValue "elem", "type"} === "button"
		){ #{next} }
		"""
	input: (next) ->
		"""
		var name = #{getName()};
		if(
		    name === "input" ||
		    name === "textarea" ||
		    name === "select" ||
		    name === "button"
		){ #{next} }
		"""
	text: (next) ->
		"""
		if(
		    #{getName()} === "input" &&
		    (
		    	!(#{hasAttrib "elem", "type"}) ||
		    	#{getAttributeValue "elem", "type"} === "text"
		    )
		){ #{next} }
		"""	
	checkbox: getAttribFunc "type", "checkbox"
	file: getAttribFunc "type", "file"
	password: getAttribFunc "type", "password"
	radio: getAttribFunc "type", "radio"
	reset: getAttribFunc "type", "reset"
	image: getAttribFunc "type", "image"
	submit: getAttribFunc "type", "submit"
}

pseudos = {}

sortByProcedure = (arr) ->
	parts = []
	i = last = 0
	j = arr.length - 1
	end = false

	while i <= j
		if procedure[arr[i].type] == -1 or (end = i == j)
			if end then i += 1
			parts = parts.concat(arr.slice(last, i).sort(({type:a}, {type:b}) ->
				procedure[a] - procedure[b]
			))
			last = parts.push arr[i] unless end
		i++

	parts

procedure =
	universal: 5
	id: 4
	tag: 3
	class: 2
	attribute: 1
	pseudo: 0
	descendant: -1
	child: -1
	sibling: -1
	adjacent: -1

parse = (selector) ->
	functions = CSSwhat(selector).map((arr) ->
		arr = sortByProcedure arr
		func = rootFunc
		i = 0
		j = arr.length
		
		while i < j && func isnt falseFunc
			func = generalRules[arr[i].type] func, arr[i]
			i++
		func
	).filter((func) -> func not in [rootFunc, falseFunc])
	
	uniqNameCounter = 0 #reset the counter

	if functions.length == 0 then falseFunc
	else if functions.length == 1 then functions[0]
	else if trueFunc in functions then trueFunc
	else functions.join "\n"

getFunc = (selector) -> new Function "elem", parse selector

CoffeeSelect = (query, elem) ->
	query = getFunc(query) if typeof query != "function"
	if elem? then CoffeeSelect.iterate query, elem else query

CoffeeSelect.parse = getFunc

CoffeeSelect.is = (elem, query) ->
	query = getFunc(query) if typeof query != "function"
	query(elem)

CoffeeSelect.iterate = (query, elems) ->
	query = getFunc(query)  unless typeof query is "function"
	return []  if query is falseFunc
	elems = elems.children  unless Array.isArray(elems)
	iterate query, elems

CoffeeSelect.filters = filters
CoffeeSelect.pseudos = pseudos


iterate = (query, elems) ->
	result = []

	for elem in elems
		continue	unless isElement(elem)
		result.push elem	if query(elem)
		result = result.concat(iterate(query, elem.children))	 if elem.children
	result

module.exports = CoffeeSelect