var expect = require('expect.js');

var $ = require('../');
var fruits = require('./fixtures').fruits;


describe('$(...)', function() {

  describe('.attr', function() {

    it('() : should get all the attributes', function() {
      var attrs = $('ul', fruits).attr();
      expect(attrs.id).to.equal('fruits');
    });

    it('(invalid key) : invalid attr should get undefined', function() {
      var attr = $('.apple', fruits).attr('lol');
      expect(attr).to.be(undefined);
    });

    it('(valid key) : valid attr should get value', function() {
      var cls = $('.apple', fruits).attr('class');
      expect(cls).to.equal('apple');
    });

    it('(key, value) : should set attr', function() {
      var $fruits = $(fruits);
      var $pear = $('.pear', $fruits).attr('id', 'pear');
      expect($('#pear', $fruits)).to.have.length(1);
      expect($pear.cheerio).to.not.be(undefined);
    });

    it('(map) : object map should set multiple attributes', function() {
      var $fruits = $(fruits);
      $('.apple', $fruits).attr({
        id: 'apple',
        style: 'color:red;',
        'data-url': 'http://apple.com'
      });
      var attrs = $('.apple', $fruits).attr();
      expect(attrs.id).to.equal('apple');
      expect(attrs.style).to.equal('color:red;');
      expect(attrs['data-url']).to.equal('http://apple.com');
    });

    it('(key, value) : should correctly encode then decode unsafe values', function() {
      var $apple = $('.apple', fruits);
      $apple.attr('href', 'http://github.com/"><script>alert("XSS!")</script><br');
      expect($apple.get(0).attribs.href).to.equal('http://github.com/&quot;&gt;&lt;script&gt;alert(&quot;XSS!&quot;)&lt;/script&gt;&lt;br');
      expect($apple.attr('href')).to.equal('http://github.com/"><script>alert("XSS!")</script><br');

      $apple.attr('href', 'http://github.com/"><script>alert("XSS!")</script><br');
      expect($apple.html()).to.not.contain('<script>alert("XSS!")</script>');
    });
  });

  describe('.removeAttr', function() {

    it('(key) : should remove a single attr', function() {
      var $fruits = $(fruits);
      expect($('ul', $fruits).attr('id')).to.not.be(undefined);
      $('ul', $fruits).removeAttr('id');
      expect($('ul', $fruits).attr('id')).to.be(undefined);
    });

    it('should return cheerio object', function() {
      var obj = $('ul', fruits).removeAttr('id').cheerio;
      expect(obj).to.be.ok();
    });

  });

  describe('.hasClass', function() {

    it('(valid class) : should return true', function() {
      var $fruits = $(fruits);
      var cls = $('.apple', $fruits).hasClass('apple');
      expect(cls).to.be.ok();
    });

    it('(invalid class) : should return false', function() {
      var cls = $('#fruits', fruits).hasClass('fruits');
      expect(cls).to.not.be.ok();
    });

    it('should check multiple classes', function() {
      var $fruits = $(fruits);

      // Add a class
      $('.apple', $fruits).addClass('red');
      expect($('.apple', $fruits).hasClass('apple')).to.be.ok();
      expect($('.apple', $fruits).hasClass('red')).to.be.ok();

      // Remove one and test again
      $('.apple', $fruits).removeClass('apple');
      expect($('li', $fruits).eq(0).hasClass('apple')).to.not.be.ok();
      expect($('li', $fruits).eq(0).hasClass('red')).to.be.ok();
    });
  });

  describe('.addClass', function() {

    it('(first class) : should add the class to the element', function() {
      var $fruits = $(fruits);
      $('#fruits', $fruits).addClass('fruits');
      var cls = $('#fruits', $fruits).hasClass('fruits');
      expect(cls).to.be.ok();
    });

    it('(single class) : should add the class to the element', function() {
      var $fruits = $(fruits);
      $('.apple', $fruits).addClass('fruit');
      var cls = $('.apple', $fruits).hasClass('fruit');
      expect(cls).to.be.ok();
    });

    it('(class): adds classes to many selected items', function() {
      var $fruits = $(fruits);
      $('li', $fruits).addClass('fruit');
      expect($('.apple', $fruits).hasClass('fruit')).to.be.ok();
      expect($('.orange', $fruits).hasClass('fruit')).to.be.ok();
      expect($('.pear', $fruits).hasClass('fruit')).to.be.ok();
    });

    it('(class class class) : should add multiple classes to the element', function() {
      var $fruits = $(fruits);
      $('.apple', $fruits).addClass('fruit red tasty');
      expect($('.apple', $fruits).hasClass('apple')).to.be.ok();
      expect($('.apple', $fruits).hasClass('fruit')).to.be.ok();
      expect($('.apple', $fruits).hasClass('red')).to.be.ok();
      expect($('.apple', $fruits).hasClass('tasty')).to.be.ok();
    });

    it('(fn) : should add classes returned from the function');

  });

  describe('.removeClass', function() {

    it('() : should remove all the classes', function() {
      var $fruits = $(fruits);
      $('.pear', $fruits).addClass('fruit');
      $('.pear', $fruits).removeClass();
      expect($('.pear', $fruits).attr('class')).to.be(undefined);
    });

    it('(invalid class) : should not remove anything', function() {
      var $fruits = $(fruits);
      $('.pear', $fruits).removeClass('fruit');
      expect($('.pear', $fruits).hasClass('pear')).to.be.ok();
    });

    it('(single class) : should remove a single class from the element', function() {
      var $fruits = $(fruits);
      $('.pear', $fruits).addClass('fruit');
      expect($('.pear', $fruits).hasClass('fruit')).to.be.ok();
      $('.pear', $fruits).removeClass('fruit');
      expect($('.pear', $fruits).hasClass('fruit')).to.not.be.ok();
      expect($('.pear', $fruits).hasClass('pear')).to.be.ok();
    });

    it('(class class class) : should remove multiple classes from the element', function() {
      var $fruits = $(fruits);

      $('.apple', $fruits).addClass('fruit red tasty');
      expect($('.apple', $fruits).hasClass('apple')).to.be.ok();
      expect($('.apple', $fruits).hasClass('fruit')).to.be.ok();
      expect($('.apple', $fruits).hasClass('red')).to.be.ok();
      expect($('.apple', $fruits).hasClass('tasty')).to.be.ok();

      $('.apple', $fruits).removeClass('apple red tasty');
      expect($('.fruit', $fruits).hasClass('apple')).to.not.be.ok();
      expect($('.fruit', $fruits).hasClass('red')).to.not.be.ok();
      expect($('.fruit', $fruits).hasClass('tasty')).to.not.be.ok();
      expect($('.fruit', $fruits).hasClass('fruit')).to.be.ok();
    });

    it('(class) : should remove all occurrences of a class name', function() {
      var $div = $('<div class="x x y x z"></div>');
      expect($div.removeClass('x').hasClass('x')).to.be(false);
    });

    it('(fn) : should remove classes returned from the function');

  });

});
