#!/usr/bin/node
/*                     PROLIFIC CONTENT MANAGEMENT SYSTEM
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= Introduction
==============

Documentation missing.

/1/pages/
    PUT - N/A
    GET - List of pages
    POST - Refresh all pages?
    DELETE - Delete all pages?
/1/pages/faq.html
    PUT - Create new page (failure if exists)
    GET - Get page data
    POST - Change page data
    DELETE - Delete page
                                                                             */
var c = require('prolific/config'),
    APIServer = require('prolific/APIServer').APIServer,
    Resource = require('prolific/Resource').Resource,
    Pages = require('./Page').Pages,
    Page = require('./Page').Page,
    syslog = require('prolific/syslog'),
    Logger = syslog.Logger,
    promisify = require('prolific/Promise').promisify,
    generateSpecialPages = require('./generate-special-pages').generateSpecialPages;

var config = c({
    programName: 'prolific-cms',
    configFile: '/etc/prolific-cms/config.json',
    options: {
        portNumber: {
            type: c.Number,
            cliName: 'port',
            defaultValue: 8080,
            help: 'TCP port number to listen for connections'
        },
        templateDir: {
            type: c.Path,
            cliName: 'template-dir',
            defaultValue: '/usr/share/prolific-cms/templates/default/',
            help: 'Template directory'
        },
        templateVariables: {
            type: c.Object,
            defaultValue: {}
        },
        specialPages: {
            type: new c.ArrayOf(new c.Object({
                name: {type: c.String},
                moduleDir: {type: c.Path}
            })),
            defaultValue: []
        },
        pagesDir: {
            type: c.Path,
            cliName: 'pages-dir',
            defaultValue: '/var/www/prolific-cms/',
            help: 'Directory to write the generated pages to'
        },
        databaseFile: {
            type: c.Path,
            cliName: 'database-file',
            defaultValue: '/var/lib/prolific-cms/database.json',
            help: 'Path to the JSON database file'
        }
    },
    shortHands: {
        d: ['--log-level', 'debug'],
        l: [
            '--config', 'config.json', '--template-dir', 'templates/default/', '--pages-dir', 'pages/',
            '--database-file', 'database.json'
        ]
    }
}, syslog.configSpec('prolific-cms'));

function interrupt() {
    syslog.log('warning', 'Interrupted, exiting');
    process.exit(1);
}

function quit() {
    try {
        server.close(function() {
            syslog.log('notice', 'All connections closed, shutting down');
            syslog.close();
            process.exit(0);
        });
    } catch (error) {
        process.exit(5);
    }
}

function exception(error) {
    try {
        syslog.log('err', 'Uncaught exception! ' + error.toString(), error);
    } catch (e) {}
    process.exit(10);
}
    
process.on('SIGINT', interrupt);
process.on('SIGTERM', interrupt);
process.on('SIGQUIT', quit);

require('./Page').init(config);

function authorize(action, id, user) {
    return true;
}
authorize = promisify(authorize);

function authenticate(request) {
    return {};
}

var pagesResource = new Resource(authorize, new Logger('Pages'), Pages),
    pageResource = new Resource(authorize, new Logger('Page'), Page),
    server = new APIServer(syslog, authenticate, [
        new RegExp('^/1/pages/$'), pagesResource,
        new RegExp('^/1/pages/([^/]+)$'), pageResource
    ]);

syslog.init(config);

generateSpecialPages(config);
server.listen(config.portNumber);

syslog.log('info', 'Prolific CMS REST API daemon initialized', config);

process.on('uncaughtException', exception);
