/*                     PROLIFIC CONTENT MANAGEMENT SYSTEM
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

Warning: db write race conditions.
                                                                             */
var promisify = require('prolific/Promise').promisify,
    Promise = require('prolific/Promise').Promise,
    urlparser = require('url'),
    http = require('http'),
    https = require('https'),
    fsp = require('./fs-promises'),
    fs = require('fs'),
    cheerio = require('cheerio'),
    Template = require('./Template').Template,
    path = require('path');

function readURL(url) {
    url = urlparser.parse(url);
    var module = {
            "http:": http,
            "https:": https
        }[url.protocol],
        response = new Promise(module.get(url), 'response');
    return processResponse(response);
}

function processResponse(response) {
    var result = new Promise(response),
        data = [];
    // FIXME: html only
    response.setEncoding('utf-8');
    response.on('data', function (chunk) {data.push(chunk);});
    response.on('end', function () {result.emit('success', data.join(''));});
    return result;
}
processResponse = promisify(processResponse);

function loadHTML(html) {
    return cheerio.load(html);
}
loadHTML = promisify(loadHTML);

function loadURL(url) {
    return loadHTML(readURL(url));
}

var template, wikiCss, pagesDir, databaseFile, db;

exports.init = function(config) {
    template = new Template(config.templateDir, config.templateVariables);
    wikiCss = fs.readFileSync(path.join(config.templateDir, 'wiki.css'));
    pagesDir = config.pagesDir;
    databaseFile = config.databaseFile;
    try {
        db = JSON.parse(fs.readFileSync(databaseFile, 'utf-8'));
    } catch (error) {
        if (error.code == 'ENOENT') {
            db = [];
        } else {
            throw error;
        }
    }
}

function generateWikiHTML($) {
    var title = $('title').html(), content = $('.qw-ftlauderdale').html();
    content = content.replace(/ id="none"/g, ''); // FIXME
    return template.generate({content: content, head: '<title>' + title + '</title>', css: wikiCss});
}
generateWikiHTML = promisify(generateWikiHTML);

function updatePage(page) {
    return fsp.writeFile(path.join(pagesDir, page.name), generateWikiHTML(loadURL(page.wikiUrl)), 'utf-8');
}

function deletePage(page) {
    return fsp.unlink('pages/' + page.name);
}

function updatePages(db) {
    var results = [];
    for (var i = 0; i < db.length; i++) {
        results.push(updatePage(db[i]));
    }
    return new Promise(results);
}

function exists(pageName) {
    for (var i = 0; i < db.length; i++) {
        if (db[i].name == pageName) return true;
    }
    return false;
}

function waitFor(promise, value) {
    return value;
}
waitFor = promisify(waitFor);

exports.Pages = {
    read: function (user, id, params) {
        var list = [];
        for (var i = 0; i < db.length; i++) {
            var page = db[i];
            list.push({name: page.name, url: page.wikiUrl});
        }
        return list.sort(function (a, b) {
            if (a.name < b.name) return -1;
            if (a.name > b.name) return 1;
            return 0;
        });
    },
    check: function (user, id, params) {
        return true;
    },
    update: function(user, id, data) {
        return waitFor(updatePages(db), 'All pages regenerated.');
    }
};

exports.Page = {
    create: function (user, id, data) {
        // FIXME: check valid page name
        // FIXME: error
        if (exists(id)) throw new Error('Already exists');
        var page = {
            type: 'page',
            name: id,
            pageType: 'wiki',
            wikiUrl: data.url
        }
        db.push(page);
        fs.writeFile(databaseFile, JSON.stringify(db), 'utf-8');
        return data.generate ? waitFor(updatePage(page), 'Page created and generated.') : 'Page created.';
    },
    read: function (user, id, params) {
        for (var i = 0; i < db.length; i++) {
            var page = db[i];
            if (page.name == id) {
                return page;
            }
        }
        // FIXME
        throw new Error('Not found');
    },
    check: function (user, id, params) {
        return exists(id);
    },
    update: function (user, id, data) {
        for (var i = 0; i < db.length; i++) {
            var page = db[i];
            if (page.name == id) {
                if (data.url) {
                    page.wikiUrl = data.url;
                    fs.writeFile(databaseFile, JSON.stringify(db), 'utf-8');
                }
                return data.regenerate ? waitFor(updatePage(page), 'Page updated and regenerated.') : 'Page updated.';
            }
        }
        // FIXME
        throw new Error('Not found');
    },
    remove: function (user, id) {
        for (var i = 0; i < db.length; i++) {
            var page = db[i];
            if (page.name == id) {
                db.splice(i, 1);
                fs.writeFile(databaseFile, JSON.stringify(db), 'utf-8');
                return waitFor(deletePage(page), 'Page deleted.');
            }
        }
        // FIXME
        throw new Error('Not found');
    }
};
