/*                         PROLIFIC UTILITIES MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= Prolific utilities
====================

This module defines a number of utility functions used by all the Prolific
modules.
                                                                             */
var util = require('util'),
    assert = require('assert');
                                                                             /*
= The define() function
=======================

The "define()" function is used to define a class of objects. It lets you
specify another "class" to inherit from, a constructor, and a list of methods
or properties. It deals with inheritance for you, and you won't have to write
"X.prototype.xxx = ..." over and over again.

Examples:

    var X = define({
        constructor: function(y) {
            this.y = y;
        }
    });

Defines a class of objects named "X". You can create instances with the usual
"var obj = new X(val)". So far, this only means more typing. (Also note that
the constructor function is optional.)

    var X = define({
        inherit: Y,
        constructor: function(z) {
            this.z = z;
        }
    });

Same as above, but inherits from Y. The constructor for Y will be called
automatically for you.

    var X = define({
        inherit: Y,
        manualSuper: true,
        constructor: function(z) {
            Y.call(this, 'Some other set of arguments');
            this.z = z;
        }
    });

The "manualSuper" option disables calling the constructor for Y automatically,
in case you need to pass a different set of arguments or anything like that.

    var X = define({
        inherit: Y,
        newMethod: function(a, b) {
            // ...
        },
        newProperty: 'etc.'
    });

You simply add any methods or properties in the definition, like that.

Let's take a look at the implementation:
                                                                             */
function define(spec) {
    assert(spec instanceof Object, 'Object expected');
    var constructor = spec.constructor,
        superConstructor = spec.inherit,
        autoCallSuper = !spec.manualSuper,
                                                                             /*
So far, nothing surprising I guess. After this, we define the actual
constructor that is returned:
                                                                             */
        Class = function(/*...*/) {
            if (autoCallSuper && superConstructor) superConstructor.apply(this, arguments);
            if (constructor) constructor.apply(this, arguments);
        };
                                                                             /*
That should hardly require any more explanations as well. We proceed with some
sanity checks, and use "util.inherits" to add "superConstructor"'s prototype to
the chain.
                                                                             */
    if (constructor) assert(constructor instanceof Function, 'constructor must be Function');
    if (superConstructor) {
        assert(superConstructor instanceof Function, 'super constructor must be Function');
        util.inherits(Class, superConstructor);
    }
                                                                             /*
Then we just have to set all the properties in the prototype object, and return
the constructor:
                                                                             */
    for (name in spec) {
        if (['constructor', 'inherit', 'manualSuper'].indexOf(name) < 0) {
            Class.prototype[name] = spec[name];
        }
    }
    return Class;
}

exports.define = define;
                                                                             /*
= The makeError() function
==========================

The "makeError()" function is used to create Error instances. Since "Error" is
somewhat special in Javascript, this is simpler and works better than trying to
inherit from "Error". You will generally use this function with "throw":

    throw makeError('MyError', 'This is a custom error');

The first argument is the error name (usually used to determine what kind of
error happened), while the second argument is a human readable message (this is
what's usually seen by the user in a stack trace for eg.).

It's also possible to pass a third argument, with extra properties to add to
the error object (eg. additional details on the error).
                                                                             */
function makeError(name, message, properties) {
    var error = new Error(message);
    error.name = name;
    if (properties) {
        for (var propName in properties) {
            error[propName] = properties[propName];
        }
    }
    return error;
}

exports.makeError = makeError;
