/*                           SIMPLE TEST FRAMEWORK
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= Testing
=========

This module allows running asynchronous tests in a very simple way.
                                                                             */
var promisify = require('./Promise').promisify,
    Promise = require('./Promise').Promise,
    assert = require('assert'),
                                                                             /*
The module is not re-entrant, however this is not a problem given the way we're
using it at this point.
                                                                             */
    tests = [];
                                                                             /*
"test()" is the main function. The test succeeds if "condition" is true,
otherwise fails. Both "condition" and "description" can be promise objects. In
that case, if a "timeout" is specified, the test automatically fails after the
given number of milliseconds (default: 20 seconds).

Example:

    test(a == b, "a == b");

The test succeeds if "a" and "b" are equal, otherwise fails.
                                                                             */
function test(condition, description, timeout) {
    var result = new Promise;
    tests.push(result);
                                                                             /*
"testCondition()" is used to resolve "condition" and "description" in case they
are promise objects.

Note that it is possible to use node's debugger to debug failed tests, there is
an automatic break point for failed tests.
                                                                             */
    testCondition(condition, description).setTimeout(timeout || 20000)
        .on('error', function(error) {
            console.log('\tTest failed: ' + description);
            console.log('\t\tbecause of: ' + error.toString());
            debugger;
            result.emit('success', 'failure');
        })
        .on('success', function(value) {result.emit('success', value);});
}

var testCondition = promisify(function(condition, description) {
    if (condition) return 'success';
    else {
        console.log('\tTest failed: ' + description);
        debugger;
        return 'failure';
    }
});
                                                                             /*
Before calling "test()" (or any other test function), however, "test.begin()"
should be called. This marks the beginnig of tests.
                                                                             */
test.begin = function(title) {
    console.log('TESTING:', title);
    tests = [];
}
                                                                             /*
At the end of tests, "test.end()" should be called. This marks the end of
tests, and allows printing a summary of the results (number of tests, number of
succeeded tests, number of failed tests).
                                                                             */
test.end = function() {
    (new Promise(tests)).on('success', function(tests) {
        var total = tests.length,
            succeded = 0,
            failed = 0;
        for (var i = 0; i < total; i++) {
            if (tests[i] == 'success') succeded++;
            else failed++;
        }
        console.log('\t', total, 'TESTS,', succeded, 'succeeded,', failed, 'failed');
        if (failed > 0) console.log('\t(try node debug ...)');
    }).throws();
}
                                                                             /*
= Shortcut test functions

The following are shortcut test functions.

    test.isFunction(myFunc, 'myFunc'); // fails if myFunc is not a function
                                                                             */
test.isFunction = function(f, name, timeout) {
    test(new Promise(function(f) {return f instanceof Function;}, [f]), 'isFunction(' + name + ')', timeout);
}
                                                                             /*
    test.isObject(obj, 'obj'); // fails if obj is not an object
                                                                             */
test.isObject = function(obj, name, timeout) {
    test(new Promise(function(obj) {return typeof obj == 'object';}, [obj]), 'isObject(' + name + ')', timeout);
}
                                                                             /*
    test.isArray(arr, 'arr'); // fails if arr is not an array
                                                                             */
test.isArray = function(arr, name, timeout) {
    test(new Promise(function(arr) {return Array.isArray(arr);}, [arr]), 'isArray(' + name + ')', timeout);
}
                                                                             /*
    test.isString(str, 'str'); // fails if str is not a string
                                                                             */
test.isString = function(str, name, timeout) {
    test(new Promise(function(str) {
        return typeof str == 'string' || str instanceof String;
    }, [str]), 'isString(' + name + ')', timeout);
}
                                                                             /*
    test.instanceOf(obj, Class, 'obj instanceof Class');
                                                                             */
test.instanceOf = function(obj, Class, description, timeout) {
    test(new Promise(function(obj) {return obj instanceof Class;}, [obj]), description, timeout);
}
                                                                             /*
    test.isEqual(a, b, 'a == b');
                                                                             */
test.isEqual = function(a, b, description, timeout) {
    test(new Promise(function(a, b) {return a == b;}, [a, b]), description, timeout);
}
                                                                             /*
    test.deepEqual(arr, [1, 2, 3], 'arr == [1, 2, 3]');
                                                                             */
test.deepEqual = function(arrA, arrB, description, timeout) {
    test(isDeepEqual(arrA, arrB), description, timeout);
}
                                                                             /*
    test.isError(value, ReferenceError, 'value');
                                                                             */
test.isError = function(value, ErrorClass, name, timeout) {
    isError(value, ErrorClass, 'isError(' + name + ', <errorName>)', timeout);
}
                                                                             /*
"test.throws()" succeeds if the given function, when called, throws the
specified kind of error.

    test.throws(myFunc, ReferenceError, 'myFunc');
                                                                             */
test.throws = function(f, ErrorClass, name) {
    var error = null;
    try {
        f();
    } catch (e) {
        error = e;
    }
    isError(error, ErrorClass, name + ' throws <errorName>');
}
                                                                             /*
The following are tests specific to promise objects. The "promise" argument
must always be a promise object.

"test.onResolution()" calls the given "f" function once the "promise" is
resolved; if "f" returns true, the test succeeds, otherwise it fails. This is
generally used to execute a test *after* a promise is resolved. (So, it's
basically like "test()", except that the test is deferred until the promise is
resolved.)

    test.onResolution(p, function() {return a == b;}, 'a == b');
                                                                             */
test.onResolution = function(promise, f, description) {
    function g() {
        test(f(), description);
    }
    promise.on('success', g);
    promise.on('error', g);
}
                                                                             /*
"test.onResoulutionThrows()" is like "test.throws()" except that the function
is not called immediately, but rather after "p" is resolved.

    test.onResolutionThrows(p, myFunc, ReferenceError, 'myFunc');
                                                                             */
test.onResolutionThrows = function(promise, f, ErrorClass, name) {
    function g() {
        test.throws(f, ErrorClass, name);
    }
    promise.on('success', g);
    promise.on('error', g);
}
                                                                             /*
We export the "test()" function directly (like the "assert" module).
                                                                             */
module.exports = test;
                                                                             /*
= Test module helper functions

"isDeepEqual()" is rather ugly, but the node.js people did not think about
exporting a isDeepEqual() function, and copy&pasting all the code from the
"assert" module here seemed like an even worse solution.
                                                                             */
function isDeepEqual(arrA, arrB) {
    try {
        assert.deepEqual(arrA, arrB);
    } catch (error) {
        return false;
    }
    return true;
}
isDeepEqual = promisify(isDeepEqual);
                                                                             /*
"isError" is a helper used by "test.isError()", "test.throws()" etc.
                                                                             */
function isError(value, ErrorClass, description, timeout) {
    var errorName;
    if (typeof ErrorClass == 'string' || ErrorClass instanceof String) {
        errorName = ErrorClass;
        ErrorClass = Error;
    } else {
        errorName = ErrorClass.prototype.name;
    }
    description = description.replace('<errorName>', errorName);
    if (value instanceof Promise) {
        var condition = new Promise;
        value.on('error', function(error) {
            condition.emit('success', (error instanceof ErrorClass) && error.name == errorName);
        });
        value.on('success', function() {condition.emit('success', false)});
        test(condition, description, timeout);
    } else {
        test((value instanceof ErrorClass) && value.name == errorName, description);
    }
}
