/*                         UNIX SYSLOG LOGGING MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= The syslog module
===================

This is simply a wrapper around the Posix syslog calls.
                                                                             */
var util = require('util'),
    posix = require('posix'),
    define = require('./utils').define,
    c = require('./config');

exports.init = function(config) {
    var logmask = {}, levels = ['emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug'],
        level = levels.indexOf(config.logLevel);
    if (level < 0) throw new TypeError('Invalid log level');
    for (; level >= 0; level--) {
        logmask[levels[level]] = true;
    }
    posix.openlog(config.logIdentity, {pid: true}, config.logFacility);
    posix.setlogmask(logmask);
}

exports.log = function(level, message, debug) {
    posix.syslog(level, level.toUpperCase() + ': ' + message);
    if (debug) posix.syslog('debug', 'DEBUG: ' + util.inspect(debug).replace(/\n[\s]*/g, ' '));
}

exports.close = function() {
    posix.close();
}
                                                                             /*
We also export a Logger class of objects that can be used with the Prolific
APIServer resources.
                                                                             */
exports.Logger = define({
    constructor: function(name) {
        this.name = name;
    },
    log: function (level, user, message, id, data) {
        if (id && typeof id == 'object' && id.length) message += ': ' + id.join('/');
        else if (typeof id == 'string' || id instanceof String) message += ': ' + id;
        exports.log(level, this.name + ' resource: ' + message, {user: user, id: id, data: data});
    }
});
                                                                             /*
The "configSpec()" can be used with the config module to get a nice config
object that can be passed to "init()".
                                                                             */
exports.configSpec = function(identity) {
    return {
        logLevel: {
            type: new c.OneOf('emerg', 'alert', 'crit', 'err', 'warning', 'notice', 'info', 'debug'),
            help: 'Log level',
            defaultValue: 'warning',
            cliName: 'log-level'
        },
        logIdentity: {
            type: c.String,
            help: 'Name to show in the log messages',
            defaultValue: identity || 'node',
            cliName: 'log-identity'
        },
        logFacility: {
            type: new c.OneOf('kern', 'user', 'mail', 'news', 'uucp', 'daemon', 'auth', 'cron', 'lpr',
                'local0', 'local1', 'local2', 'local3', 'local4', 'local5', 'local6', 'local7'),
            help: 'System log facility to use',
            defaultValue: 'daemon',
            cliName: 'log-facility'
        }
    };
}
