/*                        CYBERSOURCE SOAP API CLIENT
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= CyberSourceClient objects
===========================

"CyberSourceClient" objects represent an interface to CyberSource's SOAP API.
                                                                             */
var soap = require('soap'),
    define = require('./objects').define,
    makeError = require('./objects').makeError,
    Promise = require('./Promise').Promise,
    promisify = require('./Promise').promisify,
    version = '0.2.3',
    wsdlUrl = 'https://ics2ws.ic3.com/commerce/1.x/transactionProcessor/CyberSourceTransaction_1.26.wsdl',
    wsdlTestUrl = 'https://ics2wstest.ic3.com/commerce/1.x/transactionProcessor/CyberSourceTransaction_1.26.wsdl',
                                                                             /*
CyberSource wants to know the environment we're running on. Note that we ignore
STDERR here.
                                                                             */
    uname = new Promise(require('child_process').exec, ['uname -a']),
                                                                             /*
We prefer using names rather than numeric codes for the card type, so we define
maps between the two:
                                                                             */
    cardTypeToCSCode = {}, cSCodeToCardType = {
        '001': 'Visa',
        '002': 'MasterCard',
        '003': 'AmericanExpress',
        '033': 'VisaElectron',
        '037': 'CartaSi'
    };
                                                                             /*
"cardTypeToCSCode" is derived from "cSCodeToCardType" using the following code:
                                                                             */
for (var code in cSCodeToCardType) {
    cardTypeToCSCode[cSCodeToCardType[code].toLowerCase()] = code;
}
                                                                             /*
We only export the "CyberSourceClient" constructor:
                                                                             */
exports.CyberSourceClient = define({
    constructor: function (config) {
        this.soapClient = new Promise(soap.createClient, soap, [config.mode == 'production' ? wsdlUrl : wsdlTestUrl]);
        this.soapClient.on('success', function(client) {
            client.setSecurity(new soap.WSSecurity(config.merchantId, config.transactionKey));
        });
        this.CSMerchantId = config.merchantId;
    },
                                                                             /*
"runTransaction" is an internal method, but it can technically be used to
execute any CyberSource transaction. It does many things for you, but you still
have to know what you're doing (ie. read CyberSource's docs).
                                                                             */
    runTransaction: runTransaction,
                                                                             /*
"billOnce" can be used for a simple payment transaction.

    result = csclient.billOnce({
    });
                                                                             */
    billOnce: billOnce,
                                                                             /*
"createOnDemandSubscription" creates a payment subscription that can be used
for "on-demand" payments; that is, the subscriber is not billed automatically,
but rather you can bill them at any time using the "billSubscriber" method
below.

    result = csclient.createOnDemandSubscription({
        firstName:    'John',
        lastName:     'Doe',
        street:       '1295 Charleston Road',
        city:         'Mountain View',
        state:        'CA',
        postalCode:   '94043',
        country:      'US',
        emailAddress: 'null@cybersource.com',
        ipAddress:    '10.7.111.111'
        currency:     'USD',
        creditCard:   {
            number:          '4111111111111111',
            expirationMonth: '12',
            expirationYear:  '2020',
            cardType:        'Visa'
        },
        title:         'On-demand profile test'
    });
                                                                             */
    createOnDemandSubscription: createOnDemandSubscription,
                                                                             /*
"retrieveSubscriptionData" returns (some of) the data that was submitted using
"createOnDemandSubscription". The credit card number is masked by CyberSource
(so you can only see some of the digits).

    result = csclient.retrieveSubscriptionData(subscriptionId);

We use an extra "promisify" here to allow "subscriptionId" being a "Promise"
object.
                                                                             */
    retrieveSubscriptionData: promisify(retrieveSubscriptionData),
                                                                             /*
"updateSubscriptionData" lets you change the data associated with the
"subscriptionId" (billing address, credit card, etc.).

    result = csclient.updateSubscriptionData(subscriptionId, {
        firstName:    'John',
        lastName:     'Doe',
        street:       '1295 Charleston Road',
        city:         'Mountain View',
        state:        'CA',
        postalCode:   '94043',
        country:      'US',
        emailAddress: 'null@cybersource.com',
        ipAddress:    '10.7.111.111'
        creditCard:   {
            number:          '4111111111111111',
            expirationMonth: '12',
            expirationYear:  '2020',
            cardType:        'Visa'
        },
    });
                                                                             */
    updateSubscriptionData: promisify(updateSubscriptionData),
                                                                             /*
"billSubscriber" is used to create a payment transaction for a "on-demand"
subscription. For example, to bill USD$10.00 (by default, the currency defined
for the subscription is used):

    result = csclient.billSubscriber(subscriptionId, "10.00");
                                                                             */
    billSubscriber: promisify(billSubscriber)
});
                                                                             /*
= CyberSourceClient objects implementation

Implementation of "billOnce".
                                                                             */
function billOnce(data) {
    var items = [], item;
    for (var i = 0; i < data.items.length; i++) {
        item = {};
        for (var name in data.items[i]) {
            item[name] = data.items[i][name];
        }
        item._attrs = {id: ''+i};
        items.push(item);
    }
    return this.runTransaction({
        billTo:                data.billTo,
        item:                  items,
        purchaseTotals:        {
            currency: data.currency
        },
        card:                  data.card,
        ccAuthService:         {
            _attrs: {run: 'true'}
        },
        ccCaptureService:      {
            _attrs: {run: 'true'}
        }
    });
}
                                                                             /*
The reason we set each field manually this way is that CyberSource is picky
about the order of fields, so we have to specify everything exactly in the
order seen below; since we don't want to impose this to users of this API, we
set the correct order here, and let users use any order they want.
                                                                             */
function createOnDemandSubscription(data) {
    return this.runTransaction({
        billTo:                       {
            firstName:  data.firstName,
            lastName:   data.lastName,
            street1:    data.street,
            city:       data.city,
            state:      data.state,
            postalCode: data.postalCode,
            country:    data.country,
            email:      data.emailAddress,
            ipAddress:  data.ipAddress
        },
        purchaseTotals:               {
            currency: data.currency
        },
        card:                         {
            accountNumber:   data.creditCard.number,
            expirationMonth: data.creditCard.expirationMonth,
            expirationYear:  data.creditCard.expirationYear,
                                                                             /*
We default to "001" (which is Visa) as the card type code if an invalid card
type is passed.
                                                                             */
            cardType:        cardTypeToCSCode[data.creditCard.type.toLowerCase()] || '001'
        },
        subscription:                 {
            title:         data.title,
            paymentMethod: 'Credit card'
        },
        recurringSubscriptionInfo:    {
            frequency: 'on-demand'
        },
        paySubscriptionCreateService: {
            _attrs: {run: 'true'}
        }
                                                                             /*
Here the "getValue()" function ability to access the fields of an object result
makes things very easy. See [Promise_objects_methods] for more details.
                                                                             */
    }).getValue('paySubscriptionCreateReply', 'subscriptionID');
}
                                                                             /*
For "retrieveSubscriptionData", we use "convertSubscriptionReply" to convert
the transaction result to a similar format as what can be passed to
"createOnDemandSubscription".
                                                                             */
function retrieveSubscriptionData(subscriptionId) {
    return convertSubscriptionReply(this.runTransaction({
        recurringSubscriptionInfo:      {
            subscriptionID: subscriptionId
        },
        paySubscriptionRetrieveService: {
            _attrs: {run: 'true'}
        }
    }));
}

function convertSubscriptionReply(reply) {
    reply = reply.paySubscriptionRetrieveReply;
    return {
        firstName:    reply.firstName,
        lastName:     reply.lastName,
        street:       reply.street1,
        city:         reply.city,
        state:        reply.state,
        postalCode:   reply.postalCode,
        country:      reply.country,
        emailAddress: reply.email,
        ipAddress:    reply.ipAddress,
        currency:     reply.currency,
        creditCard:   {
            number:          reply.cardAccountNumber,
            expirationMonth: reply.cardExpirationMonth,
            expirationYear:  reply.cardExpirationYear,
                                                                             /*
We use "Unknown" if the credit card code is not supported.
                                                                             */
            type:            cSCodeToCardType[reply.cardType] || 'Unknown'
        },
        subscription: {
            status:    reply.status,
            startDate: reply.startDate,
            endDate:   reply.endDate
        }
    };
}
convertSubscriptionReply = promisify(convertSubscriptionReply);
                                                                             /*
"updateSubscription" allows updating the billing info and the credit card info
for a given subscription ID.
                                                                             */
function updateSubscriptionData(subscriptionId, data) {
    var csData = {
        billTo: {
            firstName:  data.firstName,
            lastName:   data.lastName,
            street1:    data.street,
            city:       data.city,
            state:      data.state,
            postalCode: data.postalCode,
            country:    data.country,
            email:      data.emailAddress,
            ipAddress:  data.ipAddress
        }
    };
    if (data.creditCard) {
        csData.card = {
            accountNumber:   data.creditCard.number,
            expirationMonth: data.creditCard.expirationMonth,
            expirationYear:  data.creditCard.expirationYear,
                                                                             /*
We default to "001" (which is Visa) as the card type code if an invalid card
type is passed.
                                                                             */
            cardType:        cardTypeToCSCode[data.creditCard.type.toLowerCase()] || '001'
        };
    }
    csData.recurringSubscriptionInfo = {
        subscriptionID: subscriptionId
    };
    csData.paySubscriptionUpdateService = {
        _attrs: {run: 'true'}
    };
                                                                             /*
We just want to return "true" if the data was updated, hence the "ignoreReply"
function.
                                                                             */
    return ignoreReply(this.runTransaction(csData));
}

function ignoreReply(result) {
    return true;
}
ignoreReply = promisify(ignoreReply);
                                                                             /*
"billSubscriber" uses "convertBillReply" below to turn the result into
something easier to handle.
                                                                             */
function billSubscriber(subscriptionId, amount) {
    return convertBillReply(this.runTransaction({
        purchaseTotals:            {
            //currency:         'USD',
            grandTotalAmount: amount
        },
        recurringSubscriptionInfo: {
            subscriptionID: subscriptionId
        },
        ccAuthService:             {
            _attrs: {run: 'true'}
        },
        ccCaptureService:          {
            _attrs: {run: 'true'}
        }
    }));
}

function convertBillReply(result) {
    return {
        amount:            result.ccAuthReply.amount,
        authorizationCode: result.ccAuthReply.authorizationCode,
        dateTime:          result.ccAuthReply.authorizedDateTime,
        reconciliationId:  result.ccCaptureReply.reconciliationID
    };
}
convertBillReply = promisify(convertBillReply);
                                                                             /*
"runTransaction" is the low-level method that can perform any CyberSource
transaction.
                                                                             */
function runTransaction(data) {
    return checkResult(runTransactionHelper(this.soapClient, this.CSMerchantId, uname, data));
}
                                                                             /*
It is based off "runTransactionHelper" and "checkResult":
                                                                             */
function runTransactionHelper(soapClient, merchantId, uname, otherData, callback) {
    var data = {
            merchantID:            merchantId,
            merchantReferenceCode: 'Prolific',
            clientLibrary:         'node.js SOAP',
            clientLibraryVersion:  version,
            clientEnvironment:     uname
        };
    for (var name in otherData) {
        data[name] = otherData[name];
    }
    soapClient.runTransaction(data, callback);
}
runTransactionHelper = promisify(runTransactionHelper);
                                                                             /*
Note that "checkResult" can throw errors (see the code below) in case something
goes wrong, and also that the SOAP layer may throw errors if anything goes
wrong at that level.
                                                                             */
function checkResult(result) {
    if (result.decision == 'ACCEPT') return result;
    else {
        switch (result.reasonCode) {
            case '101':
                throw new Error('Missing field');
                break;
            case '102':
                throw new Error('Invalid field');
                break;
            case '150':
                throw new Error('General system failure');
                break;
            case '151':
            case '152':
            case '250':
                throw makeError('TemporaryError', 'Server timeout');
                break;
            case '200':
            case '201':
            case '205':
            case '221':
            case '230':
                throw makeError('TransactionRejectedError', 'Need manual review');
                break;
            case '202':
                throw makeError('CardExpiredError', 'Credit card expired');
                break;
            case '203':
            case '208':
            case '209':
            case '211':
            case '220':
            case '222':
            case '231':
            case '233':
            case '240':
                throw makeError('TransactionRejectedError', 'Transaction rejected');
                break;
            case '204':
                throw makeError('TransactionRejectedError', 'Insufficent funds');
                break;
            case '210':
                throw makeError('TransactionRejectedError', 'Reached credit limit');
                break;
            case '232':
                throw makeError('TransactionRejectedError', 'Card type not supported');
                break;
            case '207':
            case '236':
                throw makeError('TemporaryError', 'Temporary error');
                break;
            case '234':
                throw new Error('Merchant account configuration error');
                break;
            default:
                throw new Error('Unrecognized CyberSource error: ' + result.decision + ' ' + result.reasonCode);
        }
    }
}
checkResult = promisify(checkResult);
