/*                       PROLIFIC CONFIGURATION MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= The configuration module
==========================

This module produces a configuration object from the command line arguments and
a configuration file, based on the given specification.
                                                                             */
var nopt = require('nopt'),
    url = require('url'),
    path = require('path'),
    define = require('./utils').define;

function c(spec/*, ...*/) {
    var cliOptsSpec = {config: path, help: Boolean}, fileOpts, all = {}, config = {};
    for (var i = 1; i < arguments.length; i++) {
        processOpts(cliOptsSpec, all, arguments[i]);
    }
    processOpts(cliOptsSpec, all, spec.options);
    cliOpts = nopt(cliOptsSpec, spec.shortHands);
    if (cliOpts.help) {
        console.log('Usage: ' + (spec.programName || process.argv[1]) + ' [options]');
        console.log('\nOptions:');
        console.log('\t--config <file>   Read options from specified file (default: ' + spec.configFile + ')');
        console.log('\t--help   Show this help text');
        for (var name in all) {
            var def = all[name], cliName = def.cliName || name, cliType = cliOptsSpec[cliName];
            if (cliType == cliType) {
                console.log('\t--' + cliName + (cliType === Boolean ? '' : ' <value>') + '   ' + def.help +
                    ('defaultValue' in def ? ' (default: ' + def.defaultValue + ')' : ''));
            }
        }
        if (Object.keys(spec.shortHands).length > 0) {
            console.log('\nShort hand options:');
            for (name in spec.shortHands) {
                console.log('\t-' + (name.length > 1 ? '-' : '') + name + '   Expands to: ' +
                    spec.shortHands[name].join(' '));
            }
        }
        process.exit(0);
    } else {
        try {
            fileOpts = require(cliOpts.config || path.resolve(spec.configFile));
        } catch (error) {
            if (error.message.match(/^Cannot find module/)) {
                fileOpts = {};
            } else {
                throw error;
            }
        }
        for (var name in all) {
            def = all[name], cliName = def.cliName || name;
            if (cliName in cliOpts) {
                config[name] = cliOpts[cliName];
            } else if (name in fileOpts && def.type.validate(config, name, fileOpts[name])) {
                // added by validate()
            } else if ('defaultValue' in def) {
                config[name] = def.defaultValue;
            } else {
                throw new Error('Missing value for configuration option "' + name + '"');
            }
        }
        return config;
    }
}

module.exports = c;

c.String = {
    validate: function(config, key, value) {
        config[key] = String(value);
        return true;
    },
    noptType: String
};

c.Boolean = {
    validate: function(config, key, value) {
        if (value instanceof Boolean) value = value.valueOf();
        else if (typeof value == 'string') {
            if (!isNaN(value)) value = !!(+value);
            else if (value == 'null' || value == 'false') value = false;
            else value = true;
        } else value = !!value;
        config[key] = value;
        return true;
    },
    noptType: Boolean
};

c.Url = {
    validate: function(config, key, value) {
        value = url.parse(String(value));
        if (value.host) {
            config[key] = value;
            return true;
        } else return false;
    },
    noptType: url
};

c.Number = {
    validate: function(config, key, value) {
        if (isNaN(value)) {
            return false;
        } else {
            config[key] = +value;
            return true;
        }
    },
    noptType: Number
};

c.Path = {
    validate: function(config, key, value) {
        config[key] = path.resolve(String(value));
        return true;
    },
    noptType: path
};

c.Date = {
    validate: function(config, key, value) {
        value = Date.parse(value);
        if (isNaN(value)) {
            return false;
        } else {
            config[key] = new Date(value);
            return true;
        }
    },
    noptType: Date
};

c.ArrayOf = define({
    constructor: function(type) {
        this.type = type;
        if ('noptType' in type) {
            this.noptType = [Array].concat(type.noptType);
        }
    },
    validate: function(config, key, value) {
        if (Array.isArray(value)) {
            config[key] = [];
            for (var i = 0; i < value.length; i++) {
                this.type.validate(config[key], config[key].length, value[i]);
            }
            return true;
        } else {
            return false;
        }
    }
});

c.OneOf = define({
    constructor: function(/* ... */) {
        this.types = arguments;
        this.noptType = [];
        for (var i = 0; i < arguments.length; i++) {
            var type = arguments[i];
            if (typeof type == 'object' && 'validate' in type) {
                if ('noptType' in type) this.noptType.push(type.noptType);
            } else this.noptType.push(type);
        }
        if (this.noptType.length == 0) delete this.noptType;
    },
    validate: function(config, key, value) {
        for (var i = 0; i < this.types.length; i++) {
            var type = this.types[i];
            if (typeof type == 'object' && 'validate' in type) {
                if (type.validate(config, key, value)) return true;
            } else if (value === type) {
                config[key] = value;
                return true;
            }
        }
        return false;
    }
});

c.Object = define({
    constructor: function(spec) {
        this.spec = spec;
    },
    validate: function(config, key, value) {
        if (typeof value == 'object') {
            var obj = {};
            for (var name in this.spec) {
                var spec = this.spec[name];
                if (!(name in value && spec.type.validate(obj, name, value[name]))) {
                    if ('defaultValue' in spec) obj[name] = spec.defaultValue;
                    else return false;
                }
            }
            config[key] = obj;
            return true;
        } else return false;
    }
});
c.Object.validate = function(config, key, value) {
    if (typeof value == 'object') {
        config[key] = value;
        return true;
    } else return false;
};

function processOpts(cliOpts, all, spec) {
    for (var name in spec) {
        var def = spec[name];
        if (def.fileOnly || !('noptType' in def.type)) {
            cliOpts[name] = NaN;
        } else {
            cliOpts[def.cliName || name] = def.type.noptType;
        }
        all[name] = def;
    }
}
