/*                          RESOURCE OBJECTS MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= Resource objects
==================

As explained in [Architecture], a Resource object represents a REST resource.
The implementation contains all the boilerplate code to handle authorization
and logging, and delegates the actual CRED methods to an underlying Data
object.

The constructor takes three values:

    new Resource(authorize, logger, data)

an "authorize()" function (see [Authorization_functions]), a Logger object (see
[Logger_objects]), and a Data object.

Resource objects have the following methods:

    .create(user, id, data)
    .read(user, id, params)
    .check(user, id, params)
    .update(user, id, data)
    .remove(user, id)

The "user" object is passed to the authorization function. "id", "data" and
"params" are passed to the data object methods. All methods return a Promise
object. (Notice that "id" is very often ignored by "create()".)

= Resource objects implementation
=================================

As usual we import stuff and define globals at the beginning of the file:
                                                                             */
var define = require('./utils').define,
    makeError = require('./utils').makeError,
    Promise = require('./Promise').Promise,
    promisify = require('./Promise').promisify,
    assert = require('assert'),
    methods = ['create', 'read', 'check', 'update', 'remove'];
                                                                             /*
Then we define the actual class of objects:
                                                                             */
exports.Resource = define({
    constructor: function(authorize, logger, data) {
        assert(authorize instanceof Function, 'expected authorization function as first argument');
        assert(logger.log instanceof Function, 'expected Logger object as second argument');
        this.authorize = authorize;
        this.logger = logger;
        this.data = data;
                                                                             /*
We automatically check the "data" object for valid methods; if a method is not
present in the "data" object, we assume that method is not valid for this
resource, and make sure it will cause the proper error.

The "validList" is used to provide the "Allow:" HTTP response header.
                                                                             */
        this.validList = [];
        for (var i = 0; i < methods.length; i++) {
            var method = methods[i];
            if (data[method] instanceof Function) {
                this.validList.push(method);
            } else {
                this[method] = mkInvalidMethod(method);
            }
        }
    },
                                                                             /*
All the boilerplate is handled by the "checkAndDoAction()" function defined
below.
                                                                             */
    create: function(user, id, data) {
        return checkAndDoAction(this, 'create', id, user, data);
    },
    read: function(user, id, params) {
        return checkAndDoAction(this, 'read', id, user, params);
    },
    check: function(user, id, params) {
        return checkAndDoAction(this, 'check', id, user, params);
    },
    update: function(user, id, data) {
        return checkAndDoAction(this, 'update', id, user, data);
    },
    remove: function(user, id) {
        return checkAndDoAction(this, 'remove', id, user, null);
    }
});
                                                                             /*
"mkInvalidMethod()" is used to replace the methods that don't have a
correspondent in the data object (see the constructor above).
                                                                             */
function mkInvalidMethod(method) {
    return function() {
        return (new Promise).errorNow(
                makeError('InvalidMethodError', 'Invalid resource method: ' + method, {validList: this.validList}));
    }
}
                                                                             /*
The "doAction()" function does all the work. "promisify()" takes care of
catching the error, and resolving "isAuthorized" (the result of the
"authorize()" function is a Promise).
                                                                             */
function doAction(resource, isAuthorized, action, id, user, data) {
    if (isAuthorized) {
        resource.logger.log('info', user, action.toUpperCase(), id, data);
        return resource.data[action].call(resource.data, user, id, data);
    } else {
        resource.logger.log('warning', user, action.toUpperCase() + ': Access denied', id);
        throw makeError('AccessDeniedError', 'Access denied');
    }
}
doAction = promisify(doAction);
                                                                             /*
"checkAndDoAction()" is thus defined trivially:
                                                                             */
function checkAndDoAction(resource, action, id, user, data) {
    return doAction(resource, resource.authorize(action, id, user), action, id, user, data);
}
