/*                           PROMISE OBJECTS MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= Promise objects
=================

This module defines the "Promise" class of objects. Such objects represent a
"promise" to return a value; that is, if you have a function that is
asynchronous, by definition it cannot return a result right away, but it will
do so at a later time, usually by calling a callback function passed as one of
the arguments; an alternative to that is returning an EventEmitter object, and
then emitting an event when with the result at a later time.

A Promise object is an EventEmitter, but also includes a number of other
features that make it easier to deal with asynchronous code.

Let's deal right away with the stuff we import from other modules:
                                                                             */
var EventEmitter = require('events').EventEmitter,
    isArray = require('util').isArray,
    assert = require('assert'),
    define = require('./utils').define,
    makeError = require('./utils').makeError;
                                                                             /*
= The Promise() constructor

There are three ways to construct a "Promise" object:

1) new Promise(f, args)
   new Promise(f, obj, args)

2) new Promise(emitter)
   new Promise(emitter, eventName)

3) new Promise([promise1, promise2, ...])

In the first case, the result is a "promise" for the result of the function "f"
when called with the arguments specified in the array "args". "f" is generally
an asynchronous function, that either takes a callback function as its last
argument, or returns an "EventEmitter" object. (Functions that return a value
immediately are also supported, in that case the resulting promise is resolved
at the next tick.)

In the callback case, the callback is assumed to accept two arguments, like
"function (error, result) { ... }". "obj", if specified, is used as the "this"
argument of the function. If either "obj" or any of the arguments in the "args"
array are promises, the function is called after they have all been resolved to
an actual value, and the actual values are passed to the function.

See [Promise_examples] for examples on how this can be useful.

In the second case, "emitter" is an instance of "EventEmitter", and it is
turned into a promise; this can be useful to use the methods of "Promise"
instances on an "EventEmitter". You can optionally specify the name of the
"success" event.

In the third case, an array of promises (or just "EventEmitter" instances) is
passed to the constructor, and the resulting promise will resolve when *all*
the promises in the array are resolved.

The implementation of the constructor should now look rather obvious:
                                                                             */
function constructPromise(/* ... */) {
    if (arguments[0] instanceof Function) {
        constructFunctionReturnPromise(this, arguments[0], arguments[1], arguments[2]);
    } else if (arguments[0] instanceof EventEmitter) {
        constructPromiseFromEmitter(this, arguments[0], arguments[1]);
    } else if (isArrayOrArguments(arguments[0])) {
        constructAggregatePromise(this, arguments[0]);
    }
    initializePromise(this);
}
                                                                             /*
= Promise objects methods

The following is the definition of "Promise" objects:
                                                                             */
var Promise = exports.Promise = define({
    inherit: EventEmitter,
    constructor: constructPromise,
                                                                             /*
The "getValue()" method returns the value of the promise if it has been
resolved, otherwise the promise itself. If any arguments are passed, the value
is assumed to be an object or an array, and the specified field or array
position is accessed; if the promise is not yet resolved, a new promise for
such a value is returned.

For example, if the value of the promise is "{a: 1}", then "getValue('a')" will
return "1" if the promise is already resolved, otherwise it will return a new
promise that will resolve to "1". It is possible to use something like
"getValue('a', 'b') to return "1" in the case of a value like "{a: {b: 1}}".
                                                                             */
    getValue: function() {
        if (this.isError) throw new ReferenceError('Access to promise value after error event');
        else if (this.isResolved) {
            if (arguments.length == 0) return this.value;
            else return dereference(this.value, args2Array(arguments));
        } else {
            if (arguments.length == 0) return this;
            else {
                var result = new Promise(), args = args2Array(arguments);
                this.on('error', function(error) {result.emit('error', error);});
                this.on('success', function(value) {result.emit('success', dereference(value, args));});
                return result;
            }
        }
    },
                                                                             /*
The "log()" method is simply a shortcut for logging (using "console.log()") the
value of the promise as soon as it is resolved.
                                                                             */
    log: function(message) {
        if (this.isResolved) console.log(message, this.getValue());
        else this.on('success', function(value) {console.log(message, value);});
        return this;
    },
                                                                             /*
The "throws()" method is simply a shortcut for setting up the promise so that
in case of an "error" event, the error object is thrown.
                                                                             */
    throws: function() {
        this.on('error', function(error) {throw error;});
        return this;
    },
                                                                             /*
The "resolveNow()" and "errorNow()" methods can be used to resolve the promise.
(They actually resolve it at the next tick.)
                                                                             */
    resolveNow: function(value) {
        var promise = this;
        process.nextTick(function() {promise.emit('success', value);});
        return promise;
    },
    errorNow: function(error) {
        var promise = this;
        process.nextTick(function() {promise.emit('error', error);});
        return promise;
    },
                                                                             /*
The "setTimeout()" method can be used to set a timeout for the Promise; if the
promise does not resolve within the specified time, a TimeoutError is caused.
                                                                             */
    setTimeout: function(time) {
        var promise = this;
        this.timeoutId = setTimeout(function() {
            promise.timeoutId = null;
            promise.emit('error', makeError('TimeoutError', 'Promise timeout reached'));
        }, time);
        function clear() {
            if (promise.timeoutId) {
                clearTimeout(promise.timeoutId);
                promise.timeoutId = null;
            }
        }
        this.on('success', clear);
        this.on('error', clear);
        return this;
    }
});
                                                                             /*
= Promise examples

    var fileContents = new Promise(fs.readFile, fs, ['/etc/passwd']);

creates a promise for the contents of the "/etc/passwd" file. Because promises
are accepted as arguments, you can then write:

    new Promise(fs.writeFile, fs, ['copy-of-passwd', fileContents]);

As soon as "fileContents" is resolved, "fs.writeFile()" is called.

    var passwd = new Promise(fs.readFile, fs, ['/etc/passwd']),
        shadow = new Promise(fs.readFile, fs, ['/etc/shadow']);

    (new Promise([passwd, shadow]))
        .on('success', function(values) {
            console.log('Both files have been read.');
        });

Creating a promise from an array of promises means that it will be resolved
only when *all* the promises in the array are resolved.

= The promisify() function
==========================

This module also exports a "promisify()" function that turns a normal
asynchronous function (or even a synchronous one) into a function that returns
a promise object, and can accept promise objects for any if its arguments.

For example, consider the following:

    var delayedAdd = promisify(function (a, b, callback) {
        setTimeout(function() {callback(null, a + b);}, 500);
    });

The above is a function that takes two numbers and returns their sum, except
that it does so after half a second. Using "promisify()" allows you to then
write something like:

    var result = delayedAdd(delayedAdd(4, 5), 6);

This will result in a promise object. After half a second, the inner
"delayedAdd()" returns 9, thus the outer "delayedAdd()" is called; that in turn
returns 15 after another half second. Without "promisify()", you'd have to
write something like:

    delayedAdd(4, 5, function (error, result) {
        delayedAdd(result, 6, function (error, result) {
            // do something with result
        }
    }

which is much less readable than the above.

Note that this also makes it much easier to write something like in the last
example in [Promise_examples]:

    fs.readFile = promisify(fs.readFile);
    var processFiles = promisify(function (passwd, shadow) {
        console.log('Both files have been read.');
    });

    processFiles(fs.readFile('/etc/passwd'), fs.readFile('/etc/shadow'));

It looks like synchronous code, but everything happens asynchronously!

"promisify()" takes an optional second argument to replace the "this" argument
for the function.
                                                                             */
exports.promisify = function(f, obj) {
    return function() {
        return new Promise(f, obj || this, arguments);
    }
}
                                                                             /*
= Promise objects implementation
================================

We'll start from the implementation of the constructor:
                                                                             */
function constructFunctionReturnPromise(promise, f, obj, args) {
    if ('undefined' == typeof args) {
        assert(isArrayOrArguments(obj), 'Expected array of arguments');
        args = isArray(obj) ? obj : args2Array(obj);
        obj = global;
    } else {
        assert(isArrayOrArguments(args), 'Expected array of arguments');
        if (!isArray(args)) args = args2Array(args);
    }
                                                                             /*
At this point we need to wait for all the values for the arguments to the
function to be "ready". Once they are, we can call the function.
                                                                             */
    var waitForAll = new Promise([obj].concat(args));
    waitForAll.on('error', function(error) {
                                                                             /*
Note that we have to check "promise.isResolved" because it's possible that a
timeout was set (using the "setTimeout()" method) on "promise", and thus it has
already been resolved at this point.
                                                                             */
        if (!promise.isResolved) {
            promise.emit('error', error);
        }
    });
    waitForAll.on('success', function(results) {
        if (!promise.isResolved) {
            callAsyncFunction(f, results, promise);
        }
    });
}
                                                                             /*
Constructing a promise from an emitter is trivial and hopefully does not need
much explanation:
                                                                             */
function constructPromiseFromEmitter(promise, emitter, eventName) {
    emitter.on('error', function(error) {promise.emit('error', error);});
    emitter.on(eventName || 'success', successCallback(promise));
}
                                                                             /*
In the case of an array of values:
                                                                             */
function constructAggregatePromise(promise, values) {
    promise.success = [];
    promise.results = [];
    function setSuccess(i) {
        return function() {
                                                                             /*
This is the callback being called when one of the promises resolves. We simply
store its result, and emit "success" for the aggregate promise if all the
values are now successful.
                                                                             */
            promise.success[i] = true;
            if (arguments.length == 1) {
                promise.results[i] = arguments[0];
            } else if (arguments.length > 1) {
                promise.results[i] = arguments;
            }
            if (allTrue(promise.success)) promise.emit('success', promise.results);
        }
    }
                                                                             /*
We cycle through all the values:
                                                                             */
    try {
        for (var i = 0; i < values.length; i++) {
            var emitter = values[i];
            if (emitter instanceof Promise && emitter.isResolved) {
                                                                             /*
If this is a promise and it is already resolved, we can just store its value.
Note that "getValue()" may throw an error if the promise resolved into an
error, hence the "try" above.
                                                                             */
                promise.success.push(true);
                promise.results[i] = emitter.getValue();
            } else if (emitter instanceof EventEmitter) {
                                                                             /*
If this is an instance of "EventEmitter" (which includes instances of
"Promise"), we set it up to store its result once it resolves. (See the
definition of "setSuccess()" above.)
                                                                             */
                promise.success.push(false);
                emitter.on('success', setSuccess(i));
                emitter.on('error', function(error) {promise.emit('error', error);});
            } else {
                                                                             /*
Otherwise this is a regular value, so we just store it:
                                                                             */
                promise.success.push(true);
                promise.results[i] = emitter;
            }
        }
                                                                             /*
If the values are already all resolved, we resolve the aggregate promise
immediately:
                                                                             */
        if (allTrue(promise.success)) promise.resolveNow(promise.results);
    } catch (error) {
                                                                             /*
In case the "getValue()" above threw an error, we resolve the aggregate right
now to an error:
                                                                             */
        promise.errorNow(error);
    }
}
                                                                             /*
This is the common initialization for all kinds of promises:
                                                                             */
function initializePromise(promise) {
    promise.isResolved = false;
    promise.value = null;
    promise.isError = false;
    promise.on('success', function(value) {
        promise.isResolved = true;
        promise.value = value;
                                                                             /*
Once the promise has been resolved, we need to "neuter" it so that it cannot
succeed or error out more than once.
                                                                             */
        neuterPromise(promise);
    });
    promise.on('error', function() {
        promise.isResolved = true;
        promise.isError = true;
        neuterPromise(promise);
    });
}
                                                                             /*
Once "neutered", calling .on() or .emit() on a promise object will cause an
error.
                                                                             */
function neuterPromise(promise) {
    promise.on = throwError;
    promise.emit = throwError;
}

function throwError() {
    throw new Error('Promise already resolved');
}
                                                                             /*
Next, we have a set of helper functions; they are used in all the code above.

"allTrue()" returns true if all the values in the passed array are true:
                                                                             */
function allTrue(arr) {
    for (var i = 0; i < arr.length; i++) {
        if (!arr[i]) return false;
    }
    return true;
}
                                                                             /*
"isArrayOrArguments()" will return true if the argument is either an Array
object, or an "arguments" object (or, actually anything that has a "length"
property).
                                                                             */
function isArrayOrArguments(arr) {
    return (arr && (typeof arr) === 'object' && (typeof arr.length) === 'number');
}
                                                                             /*
"args2Array()" simply converts an "arguments" object into a real "Array"
object.
                                                                             */
function args2Array(args) {
    return Array.prototype.slice.call(args);
}
                                                                             /*
"successCallback()" returns an event function to be used on "EventEmitter"
objects (for "emitter.on('success', ...)"); the reason for this is that the
"success" event may produce multiple values, but we want promises to have a
single value, so in the case of many arguments we simply create an array.
                                                                             */
function successCallback(promise) {
    return function() {
        if (arguments.length == 1) promise.emit('success', arguments[0]);
        else promise.emit('success', args2Array(arguments));
    };
}
                                                                             /*
"callAsyncFunction()" is used to call an asynchronous (or, even synchronous)
function:
                                                                             */
function callAsyncFunction(f, args, promise) {
    var obj = args.shift();
                                                                             /*
Remember the first element of "args" is the value to use as the "this"
argument; we pass a callback function as the last argument:
                                                                             */
    args.push(function(error, result) {
                                                                             /*
We need to check "promise.isResolved" in case the promise was already resolved
by a timeout (using the "setTimeout()" method).
                                                                             */
        if (!promise.isResolved) {
            if (error) promise.emit('error', error);
            else promise.emit('success', result);
        }
    });
    try {
                                                                             /*
We then call the function:
                                                                             */
        var result = f.apply(obj, args);
        if (result instanceof EventEmitter) {
                                                                             /*
If the result is an "EventEmitter" instance (includes promises), we bind it to
the parent promise object:
                                                                             */
            result.on('error', function(error) {promise.emit('error', error);});
            result.on('success', successCallback(promise));
        } else if (typeof result != 'undefined') {
                                                                             /*
Otherwise, if a value is returned immediately, we resolve the parent promise
object:
                                                                             */
            promise.resolveNow(result);
        }
                                                                             /*
Note that if nothing is returned, we assume that the function will call the
passed callback once complete.
                                                                             */
    } catch (error) {
                                                                             /*
In case the function throws an error, we resolve the parent promise:
                                                                             */
        promise.errorNow(error);
    }
}
                                                                             /*
"dereference()" is used by "getValue()" to get to the value of a field of an
object or a position in an array.
                                                                             */
function dereference(object, path) {
    for (var i = 0; i < path.length; i++) {
        object = object[path[i]];
    }
    return object;
}
