/*                    TEST FOR THE PROMISE OBJECTS MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

This program tests the [Promise_objects] module.
                                                                             */
var test = require('./test'),
    assert = require('assert'),
    Promise = require('./Promise.js').Promise,
    promisify = require('./Promise.js').promisify,
    EventEmitter = require('events').EventEmitter,
    define = require('./utils').define,
    makeError = require('./utils').makeError,
    Timer = define({
        inherit: EventEmitter,
        constructor: function(time, event, value) {
            var self = this;
            setTimeout(function() {self.emit(event, value);}, time);
        }
    });

function timerFunction(time, error, result, callback) {
    setTimeout(function() {callback(error, result);}, time);
}

test.begin('Promise objects');

test.isFunction(Promise, 'Promise');

var p1 = new Promise(timerFunction, [200, null, 42]);
test(typeof p1 == 'object', 'typeof p1 == object');
test.isFunction(p1.getValue, 'p1.getValue');
test.isFunction(p1.log, 'p1.log');
test.isFunction(p1.throws, 'p1.throws');
test.isFunction(p1.resolveNow, 'p1.resolveNow');
test.isFunction(p1.errorNow, 'p1.errorNow');
test.isFunction(p1.setTimeout, 'p1.setTimeout');
test.isEqual(p1, 42, 'p1 == 42');
test.onResolution(p1, function() {return p1.getValue() == 42;}, 'p1.getValue() == 42');
test.onResolutionThrows(p1, function() {return p1.on();}, Error, 'p1.on()');
test.onResolutionThrows(p1, function() {return p1.emit();}, Error, 'p1.emit()');

var p2 = new Promise(timerFunction, [200, makeError('IntentionalError'), null]);
test.isError(p2, 'IntentionalError', 'p2');
test.onResolutionThrows(p2, function() {return p2.getValue();}, ReferenceError, 'p2.getValue()');
test.onResolutionThrows(p2, function() {return p2.on();}, Error, 'p2.on()');
test.onResolutionThrows(p2, function() {return p2.emit();}, Error, 'p2.emit()');

var p3 = new Promise(timerFunction, [
                        new Promise(timerFunction, [200, null, 200]),
                        null,
                        new Promise(timerFunction, [100, null, 42])
                    ]);
test.isEqual(p3, 42, 'p3 == 42');
test.onResolution(p3, function() {return p3.getValue() == 42;}, 'p3.getValue() == 42');

var p4 = new Promise(timerFunction, [
                        new Promise(timerFunction, [200, null, 200]),
                        null,
                        new Promise(timerFunction, [100, makeError('IntentionalError'), null])
                    ]);
test.isError(p4, 'IntentionalError', 'p4');
test.onResolutionThrows(p4, function() {return p4.getValue();}, ReferenceError, 'p4.getValue()');

var p5 = new Promise(new Timer(200, 'success', 42));
test.isEqual(p5, 42, 'p5 == 42');
test.onResolution(p5, function() {return p5.getValue() == 42;}, 'p5.getValue() == 42');

var p6 = new Promise(new Timer(200, 'error', makeError('IntentionalError')));
test.isError(p6, 'IntentionalError', 'p6');
test.onResolutionThrows(p6, function() {return p6.getValue();}, ReferenceError, 'p6.getValue()');

var p7 = new Promise(new Timer(200, 'time', 42), 'time');
test.isEqual(p7, 42, 'p7 == 42');
test.onResolution(p7, function() {return p7.getValue() == 42;}, 'p7.getValue() == 42');

var p8 = new Promise([
                        new Timer(200, 'success', 42),
                        new Timer(300, 'success', 52),
                        new Timer(400, 'success', 62)
                    ]);
test.deepEqual(p8, [42, 52, 62], 'p8 == [42, 52, 62]');
test.onResolution(p8, function() {assert.deepEqual(p8.getValue(), [42, 52, 62]); return true;},
        'p8.getValue() == [42, 52, 62]');

var p9 = new Promise([
                        new Timer(200, 'success', 42),
                        new Timer(300, 'error', makeError('IntentionalError')),
                        new Timer(400, 'success', 62)
                    ]);
test.isError(p9, 'IntentionalError', 'p9');
test.onResolutionThrows(p9, function() {return p9.getValue();}, ReferenceError, 'p9.getValue()');

var p10 = (new Promise).resolveNow(42);
test.isEqual(p10, 42, 'p10 == 42');
test.onResolution(p10, function() {return p10.getValue() == 42;}, 'p10.getValue() == 42');

var p11 = (new Promise).errorNow(makeError('IntentionalError'));
test.isError(p11, 'IntentionalError', 'p11');
test.onResolutionThrows(p11, function() {return p11.getValue();}, ReferenceError, 'p11.getValue()');

var p12 = (new Promise(timerFunction, [1000, null, 42])).setTimeout(500);
test.isError(p12, 'TimeoutError', 'p12');
test.onResolutionThrows(p12, function() {return p12.getValue();}, ReferenceError, 'p12.getValue()');

var f = promisify(function(a, b) {
        return a + b;
    }),
    timerPromise = promisify(timerFunction);

var p13 = timerPromise(200, null, 42);
test.isEqual(p13, 42, 'p13 == 42')
test.onResolution(p13, function() {return p13.getValue() == 42;}, 'p13.getValue() == 42');

var p14 = f(timerPromise(200, null, 32), 10);
test.isEqual(p14, 42, 'p14 == 42')
test.onResolution(p14, function() {return p14.getValue() == 42;}, 'p14.getValue() == 42');

var p15 = f(timerPromise(200, null, 32), timerPromise(300, null, 10));
test.isEqual(p15, 42, 'p15 == 42')
test.onResolution(p15, function() {return p15.getValue() == 42;}, 'p15.getValue() == 42');

var p16 = f(timerPromise(200, makeError('IntentionalError'), null), timerPromise(300, null, 10));
test.isError(p16, 'IntentionalError', 'p16');
test.onResolutionThrows(p16, function() {return p16.getValue();}, ReferenceError, 'p16.getValue()');

var p17 = timerPromise(200, null, {a: {b: 42}}).getValue('a', 'b');
test.isEqual(p17, 42, 'p17 == 42')
test.onResolution(p17, function() {return p17.getValue() == 42;}, 'p17.getValue() == 42');

var obj1 = {name: "obj1", method: function () {return this.name;}},
    obj2 = {name: "obj2", method: promisify(obj1.method, obj1)};
test.isEqual(obj2.method(), "obj1");

test.end();
