/*                          APISERVER OBJECTS MODULE
===============================================================================
                              by Gabriele Santilli
              (C) 2012 Silent Software, Inc. All rights reserved.

= APIServer objects
===================

"APIServer" objects implement a HTTP REST API server.

    var server = new APIServer(logger, authenticate, [
        new RegExp('^/1/subscribers/([^/]*)$'),          subscriberResource,
        new RegExp('^/1/transactions/([^/]*)/([^/]*)$'), transactionResource
    ]);

The first argument to the constructor is a logger object; the second argument
is a function that takes a request object as input and returns a user object;
the third argument is an array that describes the resources to serve. If the
URL matches the regular expression, the "Resource" object that follows it is
used. The parentesized expressions in the regular expression define the value
of the "id" argument passed to the "Resource" object methods.

Given the example above, "/1/subscribers/abc" would pass the string "abc" as
the "id" to the "Resource" object methods. "/1/transactions/abc/def" would pass
the array "['abc', 'def']" as the "id".

    server.listen(8080);

Start listening on TCP port number 8080.

    server.close(callback);

Stop listening (close the TCP port); the callback is optional.
                                                                             */
var http = require('http'),
    urlparser = require('url'),
    define = require('./utils').define;

exports.APIServer = define({
    constructor: function (logger, authenticate, resources) {
        var apiServer = this;
        this.resources = resources;
        this.logger = logger;
        this.authenticate = authenticate;
        this.server = http.createServer(function(request, response) {onRequest(apiServer, request, response);});
    },
    listen: function (portNumber) {
        this.logger.log('info', 'Listening on TCP ' + portNumber);
        this.server.listen(portNumber);
    },
    close: function (callback) {
        this.logger.log('info', 'Closing TCP listen port...');
        this.server.close(callback);
    }
});
                                                                             /*
= APIServer objects implementation
==================================

Basically all the functionality is in the "onRequest" function.
                                                                             */
function onRequest(apiServer, request, response) {
    var method = request.method,
        parsed = urlparser.parse(request.url, true),
        path = parsed.pathname,
        params = parsed.query,
        data = [],
        user = apiServer.authenticate(request),
        resources = apiServer.resources,
        resource,
        logStr = request.connection.remoteAddress + ':' + request.connection.remotePort + ' ' + method + ' ' + request.url,
        debugData = {
            method: method,
            path: path,
            params: params,
            user: user
        };

    apiServer.logger.log('info', 'ACCESS: ' + logStr, debugData);
                                                                             /*
Only JSON is supported as both input and output. Note that the Content-Type
check is rather strict, we may want to relax this a bit in the future.
                                                                             */
    if (   (method == 'POST' || method == 'PUT')
        && request.headers['content-type'].toLowerCase() != 'application/json; charset=utf-8') {
                                                                             /*
We try to provide a meaningful error response. Especially because we are strict
with the value of the Content-Type, we show what we expected to see.
                                                                             */
        apiServer.logger.log('warning', 'Invalid request Content-Type: ' + request.headers['content-type']
                + ' (' + logStr + ')', debugData);
        respondNow(response, false, 415, {}, {
            error: 'Invalid Content-Type (only JSON/UTF-8 supported)',
            received: request.headers['content-type'],
            expected: 'application/json; charset=utf-8'
        });
        return;
    }
                                                                             /*
Only the standard methods are supported.
                                                                             */
    if (['PUT', 'GET', 'HEAD', 'POST', 'DELETE'].indexOf(method) < 0) {
        apiServer.logger.log('warning', 'Invalid request method: ' + logStr, debugData);
        respondNow(response, false, 400, {}, {error: 'Invalid request method'});
        return;
    }
                                                                             /*
In order to find the "Resource" object that matches the given "path" (parsed
from "request.url" above), we iterate over the "resources" array. "resourceId"
is the array of matched parentesized expressions in the RegExp, and is reduced
to a string if there is only one.
                                                                             */
    for (var i = 0; i < resources.length; i += 2) {
        var regexp = resources[i], resourceCandidate  = resources[i+1],
            resourceId = regexp.exec(path);
        if (resourceId) {
            resource = resourceCandidate;
            resourceId.shift();
            if (resourceId.length == 1) resourceId = resourceId[0];
            debugData.matched = regexp;
            debugData.resourceId = resourceId;
            break;
        }
    }
                                                                             /*
If there was no match, we provide a Not Found error response.
                                                                             */
    if (!resource) {
        apiServer.logger.log('warning', 'Resource not found: ' + logStr, debugData);
        respondNow(response, 'HEAD' == method, 404, {}, {error: 'Resource not found'});
        return;
    }

    apiServer.logger.log('debug', 'Request seems valid: ' + logStr, debugData);
                                                                             /*
Given the above checks, we can assume the data coming in is UTF-8. We simply
gather all data chunks in the "data" array. Note, that a limit check is missing
here and has to be implemented, as we don't want a client to be able to consume
all our memory.
                                                                             */
    request.setEncoding('utf8');
    request.on('data', function (chunk) {data.push(chunk);});
    request.on('end', function () {
        var resultPromise;
        try {
            if (data = data.join('')) {
                                                                             /*
The request's content (if any) is parsed as JSON.
                                                                             */
                data = JSON.parse(data);
            }
                                                                             /*
The HEAD method is somewhat special. First, the "check" method does not really
return a result, but simply a boolean value indicating whether the supplied id
exists or not. Second, HEAD is never supposed to return content, but just the
same headers that would be returned by a GET.

The other methods are handled by calling the respective method, and returning
the result value.
                                                                             */
            if (method == 'HEAD') {
                resultPromise = resource.check(user, resourceId, params)
                    .on('success', function(data) {
                        debugData.response = data;
                        apiServer.logger.log('info', 'ACCESS: ' + (data ? '200' : '404') + ' (' + logStr + ')', debugData);
                        respondNow(response, true, data ? 200 : 404, {}, null);
                    });
            } else {
                switch (method) {
                    case 'PUT':
                        resultPromise = resource.create(user, resourceId, data);
                        break;
                    case 'GET':
                        resultPromise = resource.read(user, resourceId, params);
                        break;
                    case 'POST':
                        resultPromise = resource.update(user, resourceId, data);
                        break;
                    case 'DELETE':
                        resultPromise = resource.remove(user, resourceId);
                        break;
                }
                resultPromise
                    .on('success', function(data) {
                        debugData.response = data;
                        apiServer.logger.log('info', 'ACCESS: 200 (' + logStr + ')', debugData);
                        respondNow(response, false, 200, {}, {result: data});
                    });
            }
                                                                             /*
In case of error, we try to provide a meaningful error response.
                                                                             */
            resultPromise.on('error', function(error) {
                var errorName = (error instanceof Error ? error.name : 'Unknown');
                debugData.error = error;
                switch (errorName) {
                    case 'NotFoundError':
                        apiServer.logger.log('warning', 'Resource not found: ' + logStr, debugData);
                        respondNow(response, 'HEAD' == method, 404, {}, {
                            error: 'Resource not found.'
                        });
                        break;
                    case 'AccessDeniedError':
                        apiServer.logger.log('warning', 'Access denied: ' + logStr, debugData);
                        respondNow(response, 'HEAD' == method, 403, {}, {
                            error: 'Access denied.'
                        });
                        break;
                    case 'InvalidMethodError':
                        apiServer.logger.log('warning', 'Invalid method: ' + logStr, debugData);
                        respondNow(response, 'HEAD' == method, 405, {
                            Allow: error.validList.map(function(method) {
                                return {
                                    create: 'PUT',
                                    read:   'GET',
                                    check:  'HEAD',
                                    update: 'POST',
                                    remove: 'DELETE'
                                }[method];
                            }).join(', ')
                        }, {error: error.toString()});
                        break;
                    default:
                        apiServer.logger.log('err', error.toString() + ' (' + logStr + ')', debugData);
                        respondNow(response, 'HEAD' == method, 500, {}, {error: error.toString()});
                }
            });
        } catch (error) {
                                                                             /*
We also have a "catch" for any unexpected errors in the code above.
                                                                             */
            debugData.error = error;
            apiServer.logger.log('err', error.toString() + ' (' + logStr + ')', debugData);
            respondNow(response, 'HEAD' == method, 500, {}, {error: error.toString()});
        }
    });
                                                                             /*
This part is not implemented yet; the plan is to log the case of the client
unexpectedly closing the connection, and perhaps make sure we don't try to send
a response after this.
                                                                             */
    request.on('close', function () {
    });
    response.on('close', function () {
    });
}
                                                                             /*
The "respondNow()" function is used to send a JSON response to the client.
                                                                             */
function respondNow(response, isHead, code, headers, data) {
    if (!isHead) {
        if (!(   typeof data === 'string'
              || data instanceof String)) {
            data = JSON.stringify(data);
            headers['Content-Type'] = 'application/json; charset=utf-8';
        }
        data = new Buffer(data, 'utf-8');
        headers['Content-Length'] = data.length;
    }
    response.writeHead(code, headers);
    response.end(isHead ? undefined : data);
}
